#! /usr/bin/env python3

import csv
import sys

helper = """1) Start with a spreadsheet with the following headings:

Date     Time  Callsign  Freq    Mode  RST-Sent  RST-RX  TX-Power  Name      Location
11/05/24 09:26 EI4LF     145.45  FM       599      599       5     Diarmuid  IO52NR
11/05/24 11:07 EI4ALB    145.45  FM       599      579       2.5   Simon     IO52NQ
...

2) Save as qso.csv
3) Run this program

   GNU/Linux / UNIX         ~$ chmod +x adif_gen.py
   ----------------         ~$ ./adif_gen.py

   MS Windoze               Install Python (www.python.org/downloads/windows/)
   ----------               C:> python adif_gen.py

4) A new file called "qso.adif" will appear in the directory

"""

__author__ = "Diarmuid O'Briain, EI4LF"
__program__ = "ADIF Generator"
__version__ = "1.1"

header_dict = {"date": "QSO_DATE",
               "time": "TIME_ON",
               "callsign": "CALL",
               "freq": "FREQ",
               "mode": "MODE",
               "rst sent": "RST_SENT",
               "rst received": "RST_RCVD",
               "transmit power": "TX_PWR",
               "name": "NAME",
               "location": "GRIDSQUARE"}
header = list()
data = list()
adif_headings = list()
adif_version = "3.1.4"
qso_csv = "qso.csv"
qso_adif = "qso.adif"


def date_conv(in_date):
    """Convert the date to a number only date"""
    list_ = in_date.split("/")
    if len(list_[2]) == 2:
        list_[2] = f"20{list_[2]}"
    return f"{list_[2]}{list_[1]}{list_[0]}"


def time_conv(in_time):
    """Convert the time to a number only time"""
    list_ = in_time.split(":")
    return f"{list_[0]}{list_[1]}"

# // Check for help //
if len(sys.argv) > 1:
    if sys.argv[1].lower() in ["help", "-h", "--help", "?"]:
        print(f"{__program__}\n{'*' * len(__program__)}\n")
        print(f"Author: {__author__}\nVersion: {__version__}\n")
        print(helper)
        sys.exit(1)

# // Open an ADIF filehandle for input //
with open(qso_adif, mode="w", encoding="utf-8") as adif_fh:

    # // Write ADIF Header //
    adif_fh.write(f"<ADIF_VER:{len(adif_version)}>{adif_version}\n<EOH:0>\n\n")

    # // Open CSV file as a file-handle //
    with open(qso_csv, mode="r", encoding="utf8") as csv_fh:
        csv_read = list(csv.reader(csv_fh))
        header = csv_read[0]
        data = csv_read[1:]

    # // Create headings list in ADIF format //
    for x in header:
        if x.lower() in header_dict.keys():
            adif_headings.append(header_dict[x.lower()])

    # // Loop through data rows write to ADIF file //
    for d in data:
        for c, ah in enumerate(adif_headings):
            if ah == "QSO_DATE":
                d[c] = date_conv(d[c])
            if ah == "TIME_ON":
                d[c] = time_conv(d[c])
            adif_fh.write(f"<{ah}:{len(d[c])}>{d[c]}\n")
        adif_fh.write("<EOR:0>\n\n")

print(f"'{qso_csv}' of QSOs converted to ADIF as '{qso_adif}'.\n")

sys.exit()
