#!/usr/bin/env python3
"""Pymodbus Synchronous Server (TCP)g"""

# ================================================== #
# //                 Module imports               // #
# ================================================== #

# Install the pymodbus, pyyaml and tabulate modules

import logging
import argparse
import sys
import yaml
from tabulate import tabulate # REQUIRED: pip install tabulate
from pymodbus import ModbusDeviceIdentification 
from pymodbus.server import StartTcpServer
from pymodbus.datastore import ModbusSequentialDataBlock
from pymodbus.datastore import context as context_module
from pymodbus.exceptions import ModbusException

# ================================================== #
# //        Global variable declarations          // #
# ================================================== #

__author__ = "Diarmuid Ó Briain"
__copyright__ = "Copyright 2025, SETU"
__licence__ = "European Union Public Licence v1.2"
__version__ = "2.0"

# ================================================== #
# //                   Constants                  // #
# ================================================== #

# Defaults used if none specified with -i or -p switches

MB_SVR_HOST = "127.0.0.1"
MB_SVR_PORT = 5002 
MB_CONFIG_FILE = "modbus_config.yaml"
TARGET_SIZE = 100 # Minimum size for data blocks to prevent addressing errors
DISPLAY_COUNT = 8 # Number of registers to display in the table, starting from address 0

# Modbus Register Types mapping to Pymodbus context block indices
# For reference and for the initial table display
BLOCK_TYPE_MAP = {
    1: "Coil",       # Mapped to Coil registers (0xxxx)
    2: "Discrete Input", # Mapped to Discrete Inputs (1xxxx)
    3: "Holding Register", # Mapped to Holding Registers (4xxxx)
    4: "Input Register", # Mapped to Input Registers (3xxxx)
}

# ================================================== #
# //                   Logging                    // #
# ================================================== #

# Adjust logging format slightly to make the table stand out more clearly
logging.basicConfig(format='%(levelname)s: %(message)s')

# Set up the main logger (our custom logs) to INFO
log = logging.getLogger() 
log.setLevel(logging.INFO)

# Set up the Pymodbus server logger to DEBUG to see connection and packet info
transport_log = logging.getLogger('pymodbus.server')
transport_log.setLevel(logging.DEBUG)

# ================================================== #
# //        Terminal arguements (switches)        // #
# ================================================== #

def get_commandline_arguments():
    """Reads command line options for server setup."""
    parser = argparse.ArgumentParser(description="Modbus Server")
    parser.add_argument("-i", "--ip", default=MB_SVR_HOST, 
                        help=f"IP address to bind the server to (default: {MB_SVR_HOST})")
    parser.add_argument("-p", "--port", type=int, default=MB_SVR_PORT, 
                        help=f"Port to bind the server to (default: {MB_SVR_PORT})")
    return parser.parse_args()

# ================================================== #
# //       Read and parse YAML information        // #
# ================================================== #

def load_initial_values(config_file):
    """Loads initial values and slave ID from the YAML configuration file."""
    try:
        with open(config_file, 'r') as f:
            config_data = yaml.safe_load(f)
        
        # This returns the loaded values from YAML
        return config_data.get('slave_id', 1), config_data.get('initial_values', {
            'coils': [False], 'contacts': [False],
            'holdings': [0], 'inputs': [0],
        })
    except FileNotFoundError:
        log.error(f"Configuration file '{config_file}' not found. Using defaults.")
        return 1, {'coils': [False], 'contacts': [False], 'holdings': [0], 'inputs': [0]}
    except yaml.YAMLError as e:
        log.error(f"Error parsing YAML file: {e}. Using defaults.")
        return 1, {'coils': [False], 'contacts': [False], 'holdings': [0], 'inputs': [0]}

# ================================================== #
# //   Adds extra items to list to target length  // #
# ================================================== #

def pad_list(data_list, target_size, default_value):
    """Pads a list with a default value up to the target size."""
    padded_list = list(data_list)
    if len(padded_list) < target_size:
        padded_list.extend([default_value] * (target_size - len(padded_list)))
    return padded_list

# ================================================== #
# //         Display and Logging Functions        // #
# ================================================== #

def display_MB_table_server(store_context, title):
    """Reads values directly from the Modbus store and displays them in a tabular format."""
    
    # Check if we are passing the raw initial values dict (before context creation)
    if isinstance(store_context, dict):
        # Displaying initial values, use the dict data directly
        coils_data = store_context.get('coils', list())
        contacts_data = store_context.get('contacts', list())
        holdings_data = store_context.get('holdings', list())
        inputs_data = store_context.get('inputs', list())
    
    # Passing the live ModbusDeviceContext
    else:
        # Use get_values(start_address=0, count=TARGET_SIZE) to retrieve all values
        try:
            # This relies on the internal block names (co, di, hr, ir) 
            coils_data = store_context.co.get_values(0, TARGET_SIZE) 
            contacts_data = store_context.di.get_values(0, TARGET_SIZE)
            holdings_data = store_context.hr.get_values(0, TARGET_SIZE)
            inputs_data = store_context.ir.get_values(0, TARGET_SIZE)
        except AttributeError as e:
            # Fallback check
            log.error(f"Failed to read data blocks from store: {e}")
            return
    
    # Determine the display count
    count = min(
        DISPLAY_COUNT, 
        len(coils_data), 
        len(contacts_data), 
        len(holdings_data), 
        len(inputs_data)
    )

    if count == 0:
        print("")
        print(f"--- {title} ---")
        print("No data blocks available or initial lists are empty.")
        print('-' * 65)
        return
        
    table_data = list()
    for i in range(count):
        address = i
        table_data.append([address, 
            str(coils_data[i]), 
            str(contacts_data[i]), 
            str(holdings_data[i]), 
            str(inputs_data[i])
        ])

    headers = [
        " Addr ", 
        " Coil\n(0x01, R/W) ", 
        " DI\n(0x02, R/O) ", 
        " HR\n(0x03, R/W) ", 
        " IR\n(0x04, R/O) "
    ]
    
    print("")
    print(f"--- {title} ---")
    # Output the table to stdout using print()
    print(tabulate(table_data, headers=headers, tablefmt="fancy_grid", numalign="center", stralign="center"))
    print('-' * 65)

# ================================================== #
# //             Main Server Setup                // #
# ================================================== #

def run_server(args):
    """Starts the Modbus TCP server."""
    
    # 1. Load configuration and initial data from YAML
    slave_id, INITIAL_VALUES = load_initial_values(MB_CONFIG_FILE)
    
    # 2. Display Initial State (using the raw dictionary)
    display_MB_table_server(INITIAL_VALUES, "INITIAL MODBUS STATE (Loaded from config)")

    # 3. Pad the YAML lists to ensure they meet the minimum size for addressing
    coils_padded = pad_list(INITIAL_VALUES['coils'], TARGET_SIZE, False)
    contacts_padded = pad_list(INITIAL_VALUES['contacts'], TARGET_SIZE, False)
    holdings_padded = pad_list(INITIAL_VALUES['holdings'], TARGET_SIZE, 0)
    inputs_padded = pad_list(INITIAL_VALUES['inputs'], TARGET_SIZE, 0)
    
    # 4. Create the individual Slave Context
    try:
        # Create standard blocks for all registers, allowing Pymodbus to handle writes silently.
        co_block = ModbusSequentialDataBlock(1, coils_padded)
        hr_block = ModbusSequentialDataBlock(1, holdings_padded)
        
        # Create standard blocks for read-only registers
        di_block = ModbusSequentialDataBlock(1, contacts_padded)
        ir_block = ModbusSequentialDataBlock(1, inputs_padded)

        # Create the store context using the standard ModbusDeviceContext
        store = context_module.ModbusDeviceContext( 
            co=co_block,
            di=di_block,
            hr=hr_block,
            ir=ir_block,
        )
        
    except Exception as e:
        log.critical(f"Modbus Initialisation Error: {e}")
        sys.exit(1)

    # 5. Create the central server context and identity
    context = {slave_id: store}
    
    identity = ModbusDeviceIdentification(
        info={
            0x00: 'SETU Modbus Laboratory',  # Vendor Name
            0x01: 'PM',                      # Product Code
            0x02: '1.0',                     # Major/Minor Revision
            0x03: 'Modbus Server',           # Product Name
            0x04: 'Pymodbus Lab',            # Model Name
            0x05: 'N/A',                     # Vendor URL
        }
    )
    
    log.info(f"Starting Modbus TCP server on {args.ip}:{args.port} (Unit ID: {slave_id})...")
    
    # 6. Start the server using the standard handler
    StartTcpServer(
        context=context,
        identity=identity,
        address=(args.ip, args.port),
    )

# ================================================== #
# //                Main Starter                  // #
# ================================================== #

if __name__ == "__main__":
    args = get_commandline_arguments() 
    try:
        run_server(args)
    except KeyboardInterrupt:
        log.info("Server shutting down.")
    except Exception as e:
        log.critical(f"Unhandled exception in main thread: {e}")
        sys.exit(1)
